<?php
// news_rss.php — Agregador RSS simple con cache
// Requisitos: allow_url_fopen ó cURL habilitado

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');

$q = trim($_GET['q'] ?? 'elecciones catamarca'); // texto de búsqueda
$limit = max(3, min(20, (int)($_GET['limit'] ?? 10)));
$ttl = 120; // cache en segundos

$cacheKey = 'news_' . md5($q . '|' . $limit) . '.json';
$cacheFile = sys_get_temp_dir() . DIRECTORY_SEPARATOR . $cacheKey;

// servir cache si existe y fresco
if (file_exists($cacheFile) && (time() - filemtime($cacheFile) < $ttl)) {
    readfile($cacheFile);
    exit;
}

// Fuentes base (podés sumar/quitar)
$feeds = [
    // Google News por búsqueda (muy robusto)
    'https://news.google.com/rss/search?q=' . urlencode($q) . '&hl=es-419&gl=AR&ceid=AR:es-419',
    // Ejemplos locales (si tienen RSS; dejé comentados)
    // 'https://www.elancasti.com.ar/rss', 
    // 'https://www.elesquiu.com/rss',
];

// Fetch helper
function http_get($url, $timeout = 8)
{
    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_USERAGENT => 'Catamarca-Escrutinio/1.0',
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);
        $body = curl_exec($ch);
        curl_close($ch);
        return $body ?: '';
    } else {
        $ctx = stream_context_create([
            'http' => ['timeout' => $timeout, 'header' => "User-Agent: Catamarca-Escrutinio/1.0\r\n"]
        ]);
        return @file_get_contents($url, false, $ctx) ?: '';
    }
}

// Parse helper
function parse_rss($xmlStr)
{
    $items = [];
    if (!$xmlStr) return $items;
    libxml_use_internal_errors(true);
    $xml = @simplexml_load_string($xmlStr);
    if (!$xml) return $items;

    // RSS 2.0
    if (isset($xml->channel->item)) {
        foreach ($xml->channel->item as $it) {
            $title = trim((string)$it->title);
            $link  = trim((string)$it->link);
            $date  = strtotime((string)$it->pubDate ?: (string)$it->updated ?: 'now');
            $src   = '';
            if (isset($it->source)) $src = trim((string)$it->source);
            if (!$src && isset($xml->channel->title)) $src = trim((string)$xml->channel->title);
            if ($title && $link) {
                $items[] = ['title' => $title, 'link' => $link, 'source' => $src, 'ts' => $date];
            }
        }
    } else {
        // Atom
        if (isset($xml->entry)) {
            foreach ($xml->entry as $it) {
                $title = trim((string)$it->title);
                $link  = '';
                if (isset($it->link)) {
                    foreach ($it->link as $lnk) {
                        $attrs = $lnk->attributes();
                        if (isset($attrs['href'])) {
                            $link = (string)$attrs['href'];
                            break;
                        }
                    }
                }
                $date = strtotime((string)$it->updated ?: (string)$it->published ?: 'now');
                $src  = trim((string)($xml->title ?? ''));
                if ($title && $link) {
                    $items[] = ['title' => $title, 'link' => $link, 'source' => $src, 'ts' => $date];
                }
            }
        }
    }
    return $items;
}

// Agregar y normalizar
$all = [];
foreach ($feeds as $f) {
    $raw = http_get($f);
    $items = parse_rss($raw);
    $all = array_merge($all, $items);
}

// Deduplicar por link
$tmp = [];
$dedup = [];
foreach ($all as $it) {
    $key = parse_url($it['link'], PHP_URL_PATH) ?: $it['link'];
    if (isset($tmp[$key])) continue;
    $tmp[$key] = true;
    $dedup[] = $it;
}

// Ordenar por fecha desc y cortar
usort($dedup, fn($a, $b) => ($b['ts'] ?? 0) <=> ($a['ts'] ?? 0));
$out = array_slice($dedup, 0, $limit);

// Guardar cache
$json = json_encode(['q' => $q, 'updated_at' => date('c'), 'items' => $out], JSON_UNESCAPED_UNICODE);
@file_put_contents($cacheFile, $json);

echo $json;
