<?php
require_once 'db_pdo.php';

// ====== Parámetros ======
$eleccion_id  = (int)($_GET['eleccion_id'] ?? 1);
$cargo_id     = (int)($_GET['cargo_id'] ?? 1);
$seccion_id   = (int)($_GET['seccion_id'] ?? 0);
$municipio_id = (int)($_GET['municipio_id'] ?? 0);

// ====== Datos de filtros (para mostrar arriba) ======
$eleccion = $pdo->prepare("SELECT nombre, fecha FROM eleccion WHERE id=:id");
$eleccion->execute([':id' => $eleccion_id]);
$eleccion = $eleccion->fetch();

$cargo = $pdo->prepare("SELECT nombre FROM cargo WHERE id=:id");
$cargo->execute([':id' => $cargo_id]);
$cargo = $cargo->fetch();

$seccionNombre = 'Toda la provincia';
if ($seccion_id > 0) {
    $stmt = $pdo->prepare("SELECT nombre FROM seccion WHERE id=:id");
    $stmt->execute([':id' => $seccion_id]);
    $r = $stmt->fetch();
    if ($r) $seccionNombre = $r['nombre'];
}
$municipioNombre = 'Todos';
if ($municipio_id > 0) {
    $stmt = $pdo->prepare("SELECT nombre FROM municipio WHERE id=:id");
    $stmt->execute([':id' => $municipio_id]);
    $r = $stmt->fetch();
    if ($r) $municipioNombre = $r['nombre'];
}

// ====== WHERE base ======
$whereGeo = " rpm.eleccion_id=:e AND rpm.cargo_id=:c ";
$bind     = [':e' => $eleccion_id, ':c' => $cargo_id];
if ($seccion_id > 0) {
    $whereGeo .= " AND s.id = :s ";
    $bind[':s'] = $seccion_id;
}
if ($municipio_id > 0) {
    $whereGeo .= " AND mu.id = :m ";
    $bind[':m'] = $municipio_id;
}

// ====== Votos por partido ======
$sqlPartidos = "
SELECT p.id, p.nombre, COALESCE(SUM(rpm.votos),0) AS votos
FROM resultado_partido_mesa rpm
JOIN partido p         ON p.id = rpm.partido_id
JOIN mesa m            ON m.id = rpm.mesa_id
JOIN establecimiento e ON e.id = m.establecimiento_id
JOIN circuito c        ON c.id = e.circuito_id
JOIN localidad l       ON l.id = c.localidad_id
JOIN municipio mu      ON mu.id = l.municipio_id
JOIN seccion s         ON s.id = mu.seccion_id
WHERE $whereGeo
GROUP BY p.id, p.nombre
ORDER BY votos DESC, p.nombre ASC
";
$stmt = $pdo->prepare($sqlPartidos);
$stmt->execute($bind);
$partidos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Totales/KPIs (válidos = solo partidos)
$total_validos = array_sum(array_map(fn($r) => (int)$r['votos'], $partidos));

// ====== Resumen (para blancos/nulos) ======
$sqlResumen = "
SELECT
  COALESCE(SUM(rm.votos_blanco),0)      AS blancos,
  COALESCE(SUM(rm.votos_nulos),0)       AS nulos,
  COALESCE(SUM(rm.votos_recurridos),0)  AS recurridos,
  COALESCE(SUM(rm.votos_impugnados),0)  AS impugnados,
  COALESCE(SUM(rm.electores_votaron),0) AS votaron
FROM resumen_mesa rm
JOIN mesa m            ON m.id = rm.mesa_id
JOIN establecimiento e ON e.id = m.establecimiento_id
JOIN circuito c        ON c.id = e.circuito_id
JOIN localidad l       ON l.id = c.localidad_id
JOIN municipio mu      ON mu.id = l.municipio_id
JOIN seccion s         ON s.id = mu.seccion_id
WHERE rm.eleccion_id=:e AND rm.cargo_id=:c
" . ($seccion_id > 0 ? " AND s.id = :s " : "") . ($municipio_id > 0 ? " AND mu.id = :m " : "");
$stmt = $pdo->prepare($sqlResumen);
$stmt->execute($bind);
$resumen = $stmt->fetch(PDO::FETCH_ASSOC) ?: ['blancos'=>0,'nulos'=>0,'recurridos'=>0,'impugnados'=>0,'votaron'=>0];

$total_blancos  = (int)$resumen['blancos'];
$total_nulosObs = (int)$resumen['nulos'] + (int)$resumen['recurridos'] + (int)$resumen['impugnados'];

// ====== Total emitido (nuevo denominador para % que deben sumar 100) ======
$total_general  = $total_validos + $total_blancos + $total_nulosObs;

// ====== Construcción de filas para tabla (con % del total emitido) ======
$rows = array_map(function($r) use ($total_general) {
    $v = (int)$r['votos'];
    $r['pct_total'] = $total_general > 0 ? ($v * 100.0 / $total_general) : 0.0;
    return $r;
}, $partidos);

// Agregar filas sintéticas
$rows[] = [
    'id' => 0, 'nombre' => 'Blancos',
    'votos' => $total_blancos,
    'pct_total' => $total_general > 0 ? ($total_blancos * 100.0 / $total_general) : 0.0
];
$rows[] = [
    'id' => -1, 'nombre' => 'Nulos/Obs',
    'votos' => $total_nulosObs,
    'pct_total' => $total_general > 0 ? ($total_nulosObs * 100.0 / $total_general) : 0.0
];

// ====== Heatmap tabular (ganador por sección) – opcional (sin cambios) ======
$sqlHeatmap = "
SELECT s.id AS seccion_id, s.nombre AS seccion,
       p.id AS partido_id, p.nombre AS partido, p.color,
       SUM(rpm.votos) AS votos
FROM resultado_partido_mesa rpm
JOIN partido p         ON p.id = rpm.partido_id
JOIN mesa m            ON m.id = rpm.mesa_id
JOIN establecimiento e ON e.id = m.establecimiento_id
JOIN circuito c        ON c.id = e.circuito_id
JOIN localidad l       ON l.id = c.localidad_id
JOIN municipio mu      ON mu.id = l.municipio_id
JOIN seccion s         ON s.id = mu.seccion_id
WHERE rpm.eleccion_id = :e AND rpm.cargo_id = :c
GROUP BY s.id, s.nombre, p.id, p.nombre, p.color
";
$stmt = $pdo->prepare($sqlHeatmap);
$stmt->execute([':e' => $eleccion_id, ':c' => $cargo_id]);
$rowsHeat = $stmt->fetchAll(PDO::FETCH_ASSOC);
$bySec = [];
foreach ($rowsHeat as $r) {
    $sid = (int)$r['seccion_id'];
    if (!isset($bySec[$sid])) {
        $bySec[$sid] = ['seccion' => $r['seccion'], 'total' => 0, 'gan' => ['partido' => null, 'color' => null, 'votos' => 0]];
    }
    $v = (int)$r['votos'];
    $bySec[$sid]['total'] += $v;
    if ($v > $bySec[$sid]['gan']['votos']) {
        $bySec[$sid]['gan'] = ['partido' => $r['partido'], 'color' => $r['color'], 'votos' => $v];
    }
}

// ====== Salida Excel (HTML con estilos) ======
$filename = sprintf(
    "Escrutinio_%s_%s_%s.xls",
    preg_replace('/\s+/', '', $eleccion['nombre'] ?? 'Eleccion'),
    preg_replace('/\s+/', '', $cargo['nombre'] ?? 'Cargo'),
    date('Ymd_His')
);
header("Content-Type: application/vnd.ms-excel; charset=UTF-8");
header("Content-Disposition: attachment; filename=\"$filename\"");
header("Pragma: no-cache");
header("Expires: 0");

// Helper color cálido por % (igual que en el front, pero en HEX simple)
function warmColor($pct)
{
    $t = max(0.0, min(100.0, $pct)) / 100.0;
    $a = [0xFD, 0xE6, 0x8A]; // #FDE68A
    $b = [0xDC, 0x26, 0x26]; // #DC2626
    $r = (int)round($a[0] + ($b[0] - $a[0]) * $t);
    $g = (int)round($a[1] + ($b[1] - $a[1]) * $t);
    $c = (int)round($a[2] + ($b[2] - $a[2]) * $t);
    return sprintf("#%02X%02X%02X", $r, $g, $c);
}
?>
<meta charset="utf-8" />
<style>
    body { font-family: Arial, Helvetica, sans-serif; color: #0f172a; }
    .title { font-size: 18px; font-weight: bold; margin-bottom: 6px; }
    .muted { color: #475569; font-size: 12px; }
    table { border-collapse: collapse; width: 100%; margin: 12px 0; }
    th, td { border: 1px solid #e2e8f0; padding: 6px 8px; font-size: 12px; }
    th { background: #f1f5f9; text-align: left; }
    tr:nth-child(even) td { background: #fafafa; }
    .kpis { margin-top: 8px; margin-bottom: 8px; }
    .kpis td { border: 0; padding: 2px 6px; }
    .tag { display: inline-block; padding: 2px 6px; border-radius: 6px; font-weight: bold; background: #eef2ff; color: #1e293b; }
</style>

<div class="title">Escrutinio – Catamarca</div>
<div class="muted">
    Elección: <strong><?= htmlspecialchars($eleccion['nombre'] ?? '—') ?></strong>
    (<?= htmlspecialchars($eleccion['fecha'] ?? '—') ?>) ·
    Cargo: <strong><?= htmlspecialchars($cargo['nombre'] ?? '—') ?></strong> ·
    Sección: <strong><?= htmlspecialchars($seccionNombre) ?></strong> ·
    Municipio: <strong><?= htmlspecialchars($municipioNombre) ?></strong> ·
    Exportado: <strong><?= date('d/m/Y H:i:s') ?></strong>
</div>

<table class="kpis">
    <tr>
        <td>🟢 Votos válidos:</td>
        <td><span class="tag"><?= number_format($total_validos, 0, ',', '.') ?></span></td>
        <td>⚪ Blancos:</td>
        <td><span class="tag"><?= number_format($total_blancos, 0, ',', '.') ?></span></td>
        <td>🟠 Nulos/Obs:</td>
        <td><span class="tag"><?= number_format($total_nulosObs, 0, ',', '.') ?></span></td>
        <td>🗳️ Total emitido:</td>
        <td><span class="tag"><?= number_format($total_general, 0, ',', '.') ?></span></td>
    </tr>
</table>

<table>
    <thead>
        <tr>
            <th style="width:60px;">#</th>
            <th>Partido / Categoría</th>
            <th style="width:140px;">Votos</th>
            <th style="width:140px;">% del total emitido</th>
        </tr>
    </thead>
    <tbody>
        <?php
        $rank = 1;
        foreach ($rows as $r):
        ?>
            <tr>
                <td><?= $rank++ ?></td>
                <td><?= htmlspecialchars($r['nombre']) ?></td>
                <td><?= number_format((int)$r['votos'], 0, ',', '.') ?></td>
                <td><?= number_format((float)$r['pct_total'], 2, ',', '.') ?>%</td>
            </tr>
        <?php endforeach; ?>
        <?php if (empty($rows)): ?>
            <tr><td colspan="4">Sin datos para los filtros seleccionados.</td></tr>
        <?php endif; ?>
    </tbody>
</table>

<!-- ===== Heatmap tabular (opcional, sobre válidos) ===== -->
<?php if (!empty($bySec)): ?>
    <div class="muted" style="margin-top:18px;">Ganador por Departamento (color ≈ intensidad del % sobre válidos)</div>
    <table>
        <thead>
            <tr>
                <th>Sección</th>
                <th>Ganador</th>
                <th style="width:120px;">% válidos</th>
                <th style="width:120px;">Votos gan.</th>
                <th style="width:120px;">Válidos</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($bySec as $sid => $o):
                $pct = $o['total'] > 0 ? ($o['gan']['votos'] * 100 / $o['total']) : 0;
                $bg  = warmColor($pct);
            ?>
                <tr>
                    <td><?= htmlspecialchars($o['seccion']) ?></td>
                    <td style="background:<?= $bg ?>;"><?= htmlspecialchars($o['gan']['partido'] ?? '—') ?></td>
                    <td><?= number_format($pct, 2, ',', '.') ?>%</td>
                    <td><?= number_format($o['gan']['votos'], 0, ',', '.') ?></td>
                    <td><?= number_format($o['total'], 0, ',', '.') ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>
