<?php
require_once 'db_pdo.php';

/** ====== PARAMS ====== */
$eleccion_id  = isset($_GET['eleccion_id'])  ? (int)$_GET['eleccion_id']  : 1;
$cargo_id     = isset($_GET['cargo_id'])     ? (int)$_GET['cargo_id']     : 1;
$seccion_id   = isset($_GET['seccion_id'])   ? (int)$_GET['seccion_id']   : 0;
$municipio_id = isset($_GET['municipio_id']) ? (int)$_GET['municipio_id'] : 0;

/** ====== SELECTS ====== */
$elecciones = $pdo->query("SELECT id, nombre, fecha FROM eleccion ORDER BY fecha DESC")->fetchAll();
$cargos     = $pdo->query("SELECT id, nombre FROM cargo ORDER BY id")->fetchAll();
$secciones  = $pdo->query("SELECT id, nombre FROM seccion ORDER BY nombre")->fetchAll();

/** ====== Nombres elegidos (opcionales) ====== */
$cargo_nombre_sel   = '';
$seccion_nombre_sel = '';

if ($cargo_id > 0) {
    $st = $pdo->prepare("SELECT nombre FROM cargo WHERE id = ? LIMIT 1");
    $st->execute([$cargo_id]);
    $cargo_nombre_sel = (string)$st->fetchColumn();
}
if ($seccion_id > 0) {
    $st = $pdo->prepare("SELECT nombre FROM seccion WHERE id = ? LIMIT 1");
    $st->execute([$seccion_id]);
    $seccion_nombre_sel = (string)$st->fetchColumn();
}

/** ====== ¿cpc tiene eleccion_id? ====== */
$hasEleccionCol = (bool)$pdo->query("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME   = 'circuito_partido_cargo'
      AND COLUMN_NAME  = 'eleccion_id'
")->fetchColumn();

/**
 * ===== Municipios iniciales según Sección + Cargo (regla de exclusiones) =====
 * Un municipio aparece si existe un circuito C en el municipio y existe un partido P
 * tal que NO está excluido en (C, cargo_id[, eleccion_id]) por cpc.activo=1.
 */
$municipios = [];
if ($seccion_id > 0) {
    if ($cargo_id > 0) {
        $sql = "
            SELECT DISTINCT mu.id, mu.nombre
            FROM municipio mu
            JOIN localidad  l  ON l.municipio_id = mu.id
            JOIN circuito   c  ON c.localidad_id = l.id
            WHERE mu.seccion_id = :sec
              AND EXISTS (
                    SELECT 1
                    FROM partido p
                    WHERE NOT EXISTS (
                        SELECT 1
                        FROM circuito_partido_cargo cpc
                        WHERE cpc.circuito_id = c.id
                          AND cpc.cargo_id    = :cargo
                          " . ($hasEleccionCol && $eleccion_id > 0 ? "AND cpc.eleccion_id = :elec" : "") . "
                          AND cpc.partido_id  = p.id
                          AND cpc.activo      = 1
                    )
              )
            ORDER BY mu.nombre
        ";
        $params = [':sec' => $seccion_id, ':cargo' => $cargo_id];
        if ($hasEleccionCol && $eleccion_id > 0) $params[':elec'] = $eleccion_id;

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $municipios = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } else {
        // Sin cargo: todos los municipios de la sección
        $stmt = $pdo->prepare("SELECT id, nombre FROM municipio WHERE seccion_id = :sec ORDER BY nombre");
        $stmt->execute([':sec' => $seccion_id]);
        $municipios = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

/** ====== Mensaje cuando no hay municipios ====== */
$noMunicipiosMsg = '';
if ($seccion_id > 0 && $cargo_id > 0 && empty($municipios)) {
    $cargo_min = mb_strtolower($cargo_nombre_sel ?: 'el cargo', 'UTF-8');
    $noMunicipiosMsg = "El departamento {$seccion_nombre_sel} no vota {$cargo_min}.";
}


/** ====== CORE QUERIES (para render inicial) ====== */
$filters  = [':eleccion_id' => $eleccion_id, ':cargo_id' => $cargo_id];


$cargo_nombre = '—';
foreach ($cargos as $c) {
    if ((int)$c['id'] === (int)$cargo_id) {
        $cargo_nombre = $c['nombre'];
        break;
    }
}

$whereGeo = '';
if ($seccion_id > 0) {
    $whereGeo .= " AND s.id = :seccion_id ";
    $filters[':seccion_id'] = $seccion_id;
}
if ($municipio_id > 0) {
    $whereGeo .= " AND mu.id = :municipio_id ";
    $filters[':municipio_id'] = $municipio_id;
}

$sqlPartidos = "
SELECT p.id, p.nombre, COALESCE(SUM(rpm.votos),0) AS votos
FROM resultado_partido_mesa rpm
JOIN partido p         ON p.id = rpm.partido_id
JOIN mesa m            ON m.id = rpm.mesa_id
JOIN establecimiento e ON e.id = m.establecimiento_id
JOIN circuito c        ON c.id = e.circuito_id
JOIN localidad l       ON l.id = c.localidad_id
JOIN municipio mu      ON mu.id = l.municipio_id
JOIN seccion s         ON s.id = mu.seccion_id
WHERE rpm.eleccion_id = :eleccion_id
  AND rpm.cargo_id    = :cargo_id
  $whereGeo
GROUP BY p.id, p.nombre
ORDER BY votos DESC, p.nombre ASC
";
$stmt = $pdo->prepare($sqlPartidos);
$stmt->execute($filters);
$rowsPartidos = $stmt->fetchAll();

$sqlResumen = "
SELECT
  COALESCE(SUM(rm.votos_blanco),0)      AS blancos,
  COALESCE(SUM(rm.votos_nulos),0)       AS nulos,
  COALESCE(SUM(rm.votos_recurridos),0)  AS recurridos,
  COALESCE(SUM(rm.votos_impugnados),0)  AS impugnados,
  COALESCE(SUM(rm.votos_comando),0)     AS comando,
  COALESCE(SUM(rm.electores_votaron),0) AS votaron,
  COALESCE(SUM(rm.sobres_en_urna),0)    AS sobres
FROM resumen_mesa rm
JOIN mesa m            ON m.id = rm.mesa_id
JOIN establecimiento e ON e.id = m.establecimiento_id
JOIN circuito c        ON c.id = e.circuito_id
JOIN localidad l       ON l.id = c.localidad_id
JOIN municipio mu      ON mu.id = l.municipio_id
JOIN seccion s         ON s.id = mu.seccion_id
WHERE rm.eleccion_id = :eleccion_id
  AND rm.cargo_id    = :cargo_id
  $whereGeo
";
$stmt = $pdo->prepare($sqlResumen);
$stmt->execute($filters);
$resumen = $stmt->fetch() ?: [
    'blancos' => 0,
    'nulos' => 0,
    'recurridos' => 0,
    'impugnados' => 0,
    'comando' => 0,
    'votaron' => 0,
    'sobres' => 0
];

$total_validos     = array_sum(array_map(fn($r) => (int)$r['votos'], $rowsPartidos));
$total_blancos     = (int)$resumen['blancos'];
$total_nulosObs    = (int)$resumen['nulos'] + (int)$resumen['recurridos'] + (int)$resumen['impugnados'];
// Total emitidos = válidos (partidos) + blancos + nulos/recurridos/impugnados
$total_general     = $total_validos + $total_blancos + $total_nulosObs;

$electores_votaron = (int)$resumen['votaron'];

/* ===== Porcentajes por partido (para la TABLA): % sobre total general ===== */
$rowsPartidos = array_map(function ($r) use ($total_general) {
    $v = (int)$r['votos'];
    $r['pct_partido'] = $total_general > 0 ? ($v * 100.0 / $total_general) : 0.0;
    return $r;
}, $rowsPartidos);

/* ===== KPIs (líder/segundo sobre válidos) ===== */
$lider_nombre  = $rowsPartidos[0]['nombre'] ?? '—';
$lider_votos   = isset($rowsPartidos[0]) ? (int)$rowsPartidos[0]['votos'] : 0;
$segundo_votos = isset($rowsPartidos[1]) ? (int)$rowsPartidos[1]['votos'] : 0;

$lider_pct   = $total_general > 0 ? ($lider_votos   * 100.0 / $total_general) : 0.0;
$segundo_pct = $total_general > 0 ? ($segundo_votos * 100.0 / $total_general) : 0.0;
$kpi_dif     = $lider_pct - $segundo_pct;  // ahora misma base que la tabla


/* ===== Partidos + categorías virtuales (para bar chart inicial y tabla) ===== */
/* ===== Partidos + categorías virtuales (para bar chart inicial y tabla) ===== */
$rowsPartidosPlus = $rowsPartidos;

// Fila "Blancos"
$rowsPartidosPlus[] = [
    'id'          => 0,
    'nombre'      => 'Blancos',
    'votos'       => $total_blancos,
    'pct_partido' => $total_general > 0 ? ($total_blancos * 100.0 / $total_general) : 0.0
];

// Fila "Nulos/Obs" (nulos + recurridos + impugnados)
$rowsPartidosPlus[] = [
    'id'          => -1,
    'nombre'      => 'Nulos/Obs',
    'votos'       => $total_nulosObs,
    'pct_partido' => $total_general > 0 ? ($total_nulosObs * 100.0 / $total_general) : 0.0
];

$labels = array_map(fn($r) => $r['nombre'], $rowsPartidosPlus);
$data   = array_map(fn($r) => (int)$r['votos'], $rowsPartidosPlus);

?>
<!doctype html>
<html lang="es" class="h-full">

<head>
    <meta charset="utf-8">
    <title>Escrutinio – Catamarca</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4"></script>
    <style>
        body.light {
            background: #f6f7f9;
            color: #1e2e52ff;
        }

        body.light .card {
            background: #ffffff !important;
            color: #1e2f54ff;
        }

        body.light .muted {
            color: #5e7088ff !important;
        }

        body.loading {
            cursor: progress;
        }

        .soft {
            transition: background .6s ease, transform .2s ease, color .3s ease;
        }

        .skeleton {
            position: relative;
            background: rgba(148, 163, 184, .15);
            overflow: hidden
        }



        .skeleton::after {
            content: '';
            position: absolute;
            inset: 0;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, .15), transparent);
            transform: translateX(-100%);
            animation: shine 1.2s infinite
        }

        @keyframes shine {
            to {
                transform: translateX(100%)
            }
        }

        #heatmapSecciones>div {
            transition: background .6s ease, transform .15s ease
        }

        #heatmapSecciones>div:hover {
            transform: scale(1.02)
        }

        body.light select,
        body.light input,
        body.light .bg-slate-800 {
            background: #fff !important;
            color: #0f172a !important;
            border: 1px solid #e2e8f0 !important;
        }

        body.light ::placeholder {
            color: #64748b !important
        }

        body.light select option {
            color: #0f172a !important
        }

        body.light select:focus,
        body.light input:focus {
            outline: none !important;
            border-color: #94a3b8 !important;
            box-shadow: 0 0 0 3px rgba(16, 185, 129, .25)
        }

        .btn-3d {
            position: relative;
            border: 1px solid rgba(148, 163, 184, .25);
            box-shadow: 0 8px 0 rgba(0, 0, 0, .35), 0 14px 28px rgba(0, 0, 0, .35), 0 0 0 3px rgba(16, 185, 129, .35);
            transform: translateY(0);
            transition: transform .15s, box-shadow .2s, filter .2s
        }

        .btn-3d:hover {
            filter: brightness(1.05);
            transform: translateY(3px)
        }

        .btn-3d:active {
            transform: translateY(4px)
        }

        .panel-3d {
            position: relative;
            border: 1px solid rgba(148, 163, 184, .25);
            border-radius: 16px;
            background: radial-gradient(120% 120% at 10% 0%, rgba(16, 24, 40, .35), rgba(2, 6, 23, .6)) #0b1220;
            box-shadow: 0 8px 0 rgba(0, 0, 0, .35), 0 14px 28px rgba(0, 0, 0, .35), 0 0 0 3px rgba(16, 185, 129, .18) inset;
            transition: transform .15s, box-shadow .2s, filter .2s
        }

        .panel-3d:hover {
            filter: brightness(1.04);
            transform: translateY(3px);
            box-shadow: 0 8px 0 rgba(0, 0, 0, .35), 0 14px 28px rgba(0, 0, 0, .35), 0 0 0 3px rgba(16, 185, 129, .24) inset
        }

        .panel-3d canvas {
            display: block;
            border-radius: 14px;
            box-shadow: 0 10px 24px rgba(0, 0, 0, .35);
            padding: 20px
        }

        .input-3d {
            background: rgb(2 6 23 / .9);
            border: 1px solid rgba(148, 163, 184, .28);
            color: #e5e7eb;
            border-radius: 12px;
            box-shadow: 0 6px 0 rgba(0, 0, 0, .35), 0 14px 28px rgba(0, 0, 0, .35), 0 0 0 2px rgba(16, 185, 129, .16) inset;
            transition: transform .15s, box-shadow .2s, filter .2s, background .2s
        }

        .input-3d:hover {
            filter: brightness(1.05);
            transform: translateY(2px)
        }

        .input-3d:focus {
            outline: none;
            transform: translateY(3px);
            box-shadow: 0 8px 0 rgba(0, 0, 0, .35), 0 18px 32px rgba(0, 0, 0, .40), 0 0 0 3px rgba(16, 185, 129, .35)
        }

        body.light .input-3d {
            background: #fff;
            color: #0f172a;
            border: 1px solid #e2e8f0;
            box-shadow: 0 6px 0 rgba(0, 0, 0, .12), 0 14px 28px rgba(2, 6, 23, .08), 0 0 0 2px rgba(16, 185, 129, .18) inset
        }
    </style>
</head>

<body class="bg-slate-950 text-slate-100 soft">

    <!-- HEADER -->
    <header class="sticky top-0 z-50 bg-slate-900/80 backdrop-blur border-b border-slate-800">
        <div class="max-w-7xl mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center gap-3">
                <span class="text-emerald-400 text-xl">🗳️</span>
                <div>
                    <div class="font-semibold leading-tight">Escrutinio – Catamarca</div>
                    <div class="text-xs text-slate-400">Seguimiento en vivo</div>
                </div>
            </div>

            <nav class="hidden md:flex items-center gap-2 text-sm">
                <a href="./" class="px-3 py-1 rounded-lg hover:bg-slate-800">Inicio</a>
                <a href="#analisis" class="px-3 py-1 rounded-lg hover:bg-slate-800">Análisis</a>
                <a href="#newsList" class="px-3 py-1 rounded-lg hover:bg-slate-800">Noticias</a>
            </nav>

            <div class="flex items-center gap-2">
                <!-- 🔎 Buscador por Mesa -->
                <form action="buscar_mesa.php" method="get" class="hidden sm:flex items-center gap-2">
                    <input type="hidden" name="eleccion_id" value="<?= (int)$eleccion_id ?>">
                    <input type="hidden" name="cargo_id" value="<?= (int)$cargo_id ?>">
                    <input
                        name="mesa"
                        inputmode="numeric"
                        pattern="\d+"
                        placeholder="Mesa #"
                        class="bg-slate-800/90 border border-slate-700 rounded-lg px-2 py-1 text-xs focus:outline-none focus:ring-2 focus:ring-emerald-600"
                        title="Ingresá el número de mesa"
                        required />
                    <button class="px-2 py-1 rounded-lg bg-emerald-600 hover:bg-emerald-500 text-xs font-semibold">
                        Buscar
                    </button>
                </form>

                <button id="toggleVoz" class="px-3 py-1 rounded-lg bg-fuchsia-600/20 hover:bg-fuchsia-600/30 border border-fuchsia-600/40 text-xs">
                    🔊 Voz: <span id="toggleVozState">ON</span>
                </button>
                <button id="toggleThemeTop" class="px-3 py-1 rounded-lg bg-slate-800 text-xs" title="Modo claro/oscuro">☀️ / 🌙</button>
            </div>
        </div>
    </header>


    <div class="max-w-7xl mx-auto p-4 space-y-6">

        <!-- Filtros -->
        <header class="flex flex-col md:flex-row md:items-end md:justify-between gap-4">
            <div class="flex items-center gap-3">
                <button id="toggleTheme" class="px-3 py-1 rounded-lg bg-slate-800 text-sm soft mb-5" title="Modo claro/oscuro">☀️ / 🌙</button>
            </div>
            <form id="filtrosForm" class="grid grid-cols-1 md:grid-cols-5 gap-3 card bg-slate-800/100 p-3 rounded-xl soft" method="get">
                <!-- Elección -->
                <label class="text-sm">
                    <span class="block mb-1 muted">Elección</span>
                    <select name="eleccion_id" id="eleccion" class="w-full bg-slate-900 rounded-lg p-2">
                        <?php foreach ($elecciones as $e): ?>
                            <option value="<?= $e['id'] ?>" <?= $e['id'] == $eleccion_id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($e['nombre']) . ' (' . $e['fecha'] . ')' ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>

                <!-- Departamento (Sección) -->
                <label class="text-sm">
                    <span class="block mb-1 muted">Departamento (Sección)</span>
                    <select name="seccion_id" id="seccion" class="w-full bg-slate-900 rounded-lg p-2">
                        <option value="0">Toda la provincia</option>
                        <?php foreach ($secciones as $s): ?>
                            <option value="<?= $s['id'] ?>" <?= $s['id'] == $seccion_id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($s['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>

                <!-- Cargo -->
                <label class="text-sm">
                    <span class="block mb-1 muted">Cargo</span>
                    <select name="cargo_id" id="cargo" class="w-full bg-slate-900 rounded-lg p-2">
                        <?php foreach ($cargos as $c): ?>
                            <option value="<?= $c['id'] ?>" <?= $c['id'] == $cargo_id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($c['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>

                <!-- Municipio (dinámico por sección + cargo + elección) -->
                <label class="text-sm">
                    <span class="block mb-1 muted">Municipio</span>
                    <select name="municipio_id" id="municipio" class="w-full bg-slate-900 rounded-lg p-2">
                        <option value="0">Todos</option>
                        <?php foreach ($municipios as $m): ?>
                            <option value="<?= $m['id'] ?>" <?= $m['id'] == $municipio_id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($m['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>

                <!-- Mensaje -->
                <div id="noMunicipiosMsg"
                    class="col-span-full text-sm text-amber-400 font-medium <?= $noMunicipiosMsg === '' ? 'hidden' : '' ?>">
                    <?= htmlspecialchars($noMunicipiosMsg) ?>
                </div>

                <div class="flex items-end">
                    <button class="w-full bg-emerald-600 hover:bg-emerald-500 transition rounded-lg px-4 py-2 font-semibold">
                        Aplicar filtros
                    </button>
                </div>
            </form>


        </header>

        <!-- Resumen cabecera -->
        <section class="card bg-slate-800/100 rounded-2xl p-3 flex flex-col md:flex-row items-start md:items-center justify-between gap-3 soft">

            <div class="text-xs text-emerald-400 opacity-80">🕐 Última actualización: <span id="lastUpdate">—</span></div>
        </section>

        <!-- KPIs -->


        <!-- KPIs secundarios -->
        <section class="grid grid-cols-2 md:grid-cols-4 gap-3">
            <!-- Líder -->
            <div class="card bg-slate-800/100 rounded-2xl p-4 soft">
                <div class="muted text-xs">Líder</div>
                <div class="text-base md:text-xl font-semibold"><?= htmlspecialchars($lider_nombre) ?></div>
                <div class="text-sm opacity-80"><?= number_format($lider_pct, 2, ',', '.') ?>% votos</div>

            </div>

            <!-- Departamento (Sección) -->
            <div class="card bg-slate-800/100 rounded-2xl p-4 soft">
                <div class="muted text-xs">Departamento</div>
                <div class="text-base md:text-lg font-semibold">
                    <?= $seccion_id > 0
                        ? htmlspecialchars($pdo->query("SELECT nombre FROM seccion WHERE id = $seccion_id")->fetchColumn() ?: 'Toda la provincia')
                        : 'Toda la provincia' ?>
                </div>
            </div>

            <!-- Municipio -->
            <div class="card bg-slate-800/100 rounded-2xl p-4 soft">
                <div class="muted text-xs">Municipio</div>
                <div class="text-base md:text-lg font-semibold">
                    <?php
                    if ($municipio_id > 0) {
                        $mun_nombre = $pdo->query("SELECT nombre FROM municipio WHERE id = $municipio_id")->fetchColumn();
                        echo htmlspecialchars($mun_nombre ?: 'Todos');
                    } else {
                        echo 'Todos';
                    }
                    ?>
                </div>
            </div>
        </section>


        <!-- Tabla por partido -->
        <section class="card bg-slate-800/100 rounded-2xl p-4 soft">
            <div class="flex items-center justify-between mb-3">
                <h2 class="font-semibold">Detalle por partido – <?= htmlspecialchars($cargo_nombre) ?></h2>
                <input id="searchPartido" placeholder="Buscar partido..." class="bg-slate-900 rounded p-2 text-sm w-60 soft" />
            </div>
            <div class="overflow-x-auto">
                <table class="min-w-full text-sm">
                    <thead class="text-left muted">
                        <tr>
                            <th class="py-2">#</th>
                            <th class="py-2">Partido</th>
                            <th class="py-2">Votos</th>
                            <th class="py-2">% votos</th>
                        </tr>
                    </thead>
                    <tbody id="tablaPartidos">
                        <?php $rank = 1;
                        foreach ($rowsPartidosPlus as $r):
                            $isVirtual = ((int)$r['id'] <= 0); ?>
                            <tr
                                <?php if (!$isVirtual): ?>
                                role="button" tabindex="0"
                                class="border-t border-slate-900 cursor-pointer hover:bg-slate-900/40 focus:outline-none focus:ring-2 focus:ring-emerald-600"
                                data-partido-id="<?= (int)$r['id'] ?>"
                                onclick="openDrill(<?= (int)$r['id'] ?>, <?= json_encode($r['nombre'], JSON_UNESCAPED_UNICODE) ?>)"
                                <?php else: ?>
                                class="border-t border-slate-900 opacity-90"
                                <?php endif; ?>>
                                <td class="py-2 pr-4"><?= $rank++ ?></td>
                                <td class="py-2 pr-4"><?= htmlspecialchars($r['nombre']) ?></td>
                                <td class="py-2 pr-4"><?= number_format($r['votos'], 0, ',', '.') ?></td>
                                <td class="py-2 pr-4"><?= number_format($r['pct_partido'], 2, ',', '.') ?>%</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </section>

        <!-- Charts -->
        <section class="flex flex-col gap-6">

            <!-- Toggle -->
            <div class="flex items-center justify-end">
                <div class="inline-flex rounded-xl overflow-hidden border border-slate-700">
                    <button id="btnChartBar"
                        class="px-3 py-1.5 text-xs bg-emerald-600/20 hover:bg-emerald-600/30"
                        aria-pressed="true"
                        onclick="toggleChart('bar')">
                        Votos por partido
                    </button>
                    <button id="btnChartPie"
                        class="px-3 py-1.5 text-xs hover:bg-slate-800"
                        aria-pressed="false"
                        onclick="toggleChart('pie')">
                        Composición (%)
                    </button>
                </div>
            </div>

            <!-- BAR: visible por defecto -->
            <div id="chartsBar" class="lg:col-span-1 card bg-slate-800/100 rounded-2xl p-4 soft">
                <div class="flex items-center justify-between mb-3">
                    <h2 class="font-semibold">
                        Votos por partido – <?= htmlspecialchars($cargo_nombre) ?>
                        <span class="muted font-normal">(incluye Blancos y Nulos/Obs)</span>
                    </h2>
                    <button onclick="exportCSV()" class="text-xs px-3 py-1 rounded-lg bg-slate-800 hover:bg-slate-700">📤 Exportar CSV</button>
                </div>
                <div id="chartBarWrap" class="relative">
                    <canvas id="chartBar" height="120"></canvas>
                    <div id="barSkeleton" class="absolute inset-0 skeleton rounded-xl hidden"></div>
                </div>
            </div>

            <!-- PIE: oculto al inicio -->
            <div id="chartsPie" class="card bg-slate-800/100 rounded-2xl p-4 soft composicici hidden">
                <h2 class="font-semibold mb-3">
                    Composición por partido (%) – <?= htmlspecialchars($cargo_nombre) ?>
                </h2>
                <div id="chartPieWrap" class="relative">
                    <canvas id="chartPie" height="120"></canvas>
                    <div id="pieSkeleton" class="absolute inset-0 skeleton rounded-xl hidden"></div>
                </div>
            </div>
        </section>

        <!-- Heatmap por Sección -->
        <section class="card bg-slate-800/100 rounded-2xl p-4 soft">
            <div class="flex items-center justify-between mb-3">
                <h2 id="heatmapTitle" class="font-semibold">
                    % de ganador por Departamento – <?= htmlspecialchars($cargo_nombre) ?>
                </h2>
                <div class="text-xs muted">Tip: clic en tarjeta para filtrar por sección</div>
            </div>
            <div id="heatmapSecciones" class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-3"></div>
        </section>




        <!-- Noticias en vivo -->
        <section class="card bg-slate-800/100 rounded-2xl p-4 soft">
            <div class="flex items-center justify-between mb-3">
                <h2 class="font-semibold">📰 Noticias en vivo</h2>
                <div class="text-xs muted"> <label class="mr-2">Búsqueda:</label> <input id="newsQuery" class="bg-slate-900 rounded px-2 py-1 text-xs w-48" placeholder="ej: elecciones catamarca"> <button id="newsBtn" class="ml-2 bg-emerald-600 hover:bg-emerald-500 text-xs px-3 py-1 rounded">Buscar</button> </div>
            </div>
            <div id="newsTicker" class="overflow-hidden whitespace-nowrap text-xs py-2 px-3 rounded bg-slate-900/50 mb-3 soft" style="mask-image: linear-gradient(90deg, transparent, black 10%, black 90%, transparent);">
                <div id="newsTickerInner" class="inline-block will-change-transform"></div>
            </div>
            <div id="newsList" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3 text-sm"></div>
            <div class="text-xs opacity-70 mt-2">Última actualización: <span id="newsUpdated">—</span></div>
        </section>
    </div>






    <footer class="mt-10 border-t border-slate-800 bg-slate-950">
        <div class="max-w-7xl mx-auto px-4 py-6 text-sm text-slate-400 grid gap-2 md:flex md:items-center md:justify-between">
            <div>
                <div>© <?= date('Y') ?> Escrutinio Catamarca</div>
                <div class="text-xs opacity-70">Última actualización: <span id="lastUpdateFooter">—</span></div>
            </div>
            <div class="flex items-center gap-3">
                <a href="#top" class="px-3 py-1 rounded-lg bg-slate-800 hover:bg-slate-700">Volver arriba</a>
                <a href="#newsList" class="px-3 py-1 rounded-lg bg-slate-800 hover:bg-slate-700">Noticias</a>
                <a href="#analisis" class="px-3 py-1 rounded-lg bg-slate-800 hover:bg-slate-700">Análisis</a>
            </div>
        </div>
    </footer>

    <!-- Modal drill -->
    <div id="drillModal" class="fixed inset-0 bg-black/60 hidden items-center justify-center p-4">
        <div class="bg-slate-900 rounded-2xl max-w-3xl w-full p-4">
            <div class="flex items-center justify-between mb-3">
                <h3 class="font-semibold">Detalle del partido</h3>
                <button onclick="closeDrill()" class="px-3 py-1 bg-slate-800 rounded">Cerrar</button>
            </div>
            <div id="drillBody" class="text-sm"></div>
        </div>
    </div>

    <script>
        /* ===== Estado inicial PHP → JS ===== */
        const labelsPHP = <?= json_encode($labels, JSON_UNESCAPED_UNICODE) ?>;
        const dataPHP = <?= json_encode($data) ?>;
        const totalValidosPHP = <?= (int)$total_validos ?>;
        const blancosPHP = <?= (int)$total_blancos ?>;
        const nulosObsPHP = <?= (int)$total_nulosObs ?>;

        /* ===== Tema ===== */
        document.getElementById('toggleTheme').addEventListener('click', () => document.body.classList.toggle('light'));
        document.getElementById('toggleThemeTop')?.addEventListener('click', () => document.body.classList.toggle('light'));

        /* ===== Chart plugins ===== */
        const chartShadowPlugin = {
            id: 'shadow3d',
            beforeDatasetsDraw(chart) {
                const {
                    ctx
                } = chart;
                ctx.save();
                ctx.shadowColor = 'rgba(0,0,0,.45)';
                ctx.shadowBlur = 24;
                ctx.shadowOffsetY = 12;
            },
            afterDatasetsDraw(chart) {
                chart.ctx.restore();
            }
        };
        Chart.register(chartShadowPlugin);

        function withAlpha(base, a) {
            if (typeof base !== 'string') return `rgba(0,0,0,${a})`;
            if (base.startsWith('rgba')) return base.replace(/rgba\(([^)]+)\)/, (m, i) => {
                const p = i.split(',').map(v => v.trim());
                p[3] = a.toString();
                return `rgba(${p.join(',')})`;
            });
            if (base.startsWith('rgb(')) {
                const i = base.slice(4, -1);
                return `rgba(${i}, ${a})`;
            }
            return `rgba(0,0,0,${a})`;
        }

        const extrude3DPlugin = {
            id: 'extrude3d',
            afterDatasetsDraw(chart, _, opts) {
                if (chart.config.type === 'bar') extrudeBars(chart, opts);
            },
            beforeDatasetsDraw(chart, _, opts) {
                if (chart.config.type === 'pie' || chart.config.type === 'doughnut') extrudeDonut(chart, opts);
            },
            defaults: {
                depth: 10,
                tilt: 6,
                rightFaceAlpha: .22,
                topFaceAlpha: .10,
                sideAlpha: .25
            }
        };
        Chart.register(extrude3DPlugin);

        function extrudeBars(chart, opts) {
            const {
                ctx
            } = chart;
            const depth = opts.depth ?? 10;
            const tilt = opts.tilt ?? 6;
            chart.data.datasets.forEach((ds, di) => {
                const meta = chart.getDatasetMeta(di);
                meta.data.forEach(bar => {
                    const p = bar.getProps(['x', 'y', 'base', 'width'], true);
                    const left = p.x - p.width / 2,
                        right = p.x + p.width / 2,
                        topY = p.y,
                        baseY = p.base;
                    const frontColor = Array.isArray(ds.backgroundColor) ? ds.backgroundColor[0] : ds.backgroundColor;
                    const dx = depth,
                        dy = tilt;
                    const topColor = withAlpha(frontColor, opts.topFaceAlpha ?? .10);
                    const rightColor = withAlpha(frontColor, opts.rightFaceAlpha ?? .22);
                    ctx.save();
                    ctx.beginPath();
                    ctx.moveTo(left, topY);
                    ctx.lineTo(right, topY);
                    ctx.lineTo(right + dx, topY - dy);
                    ctx.lineTo(left + dx, topY - dy);
                    ctx.closePath();
                    ctx.fillStyle = topColor;
                    ctx.fill();
                    ctx.beginPath();
                    ctx.moveTo(right, topY);
                    ctx.lineTo(right, baseY);
                    ctx.lineTo(right + dx, baseY - dy);
                    ctx.lineTo(right + dx, topY - dy);
                    ctx.closePath();
                    ctx.fillStyle = rightColor;
                    ctx.fill();
                    ctx.restore();
                });
            });
        }

        function extrudeDonut(chart, opts) {
            const {
                ctx
            } = chart;
            const depth = opts.depth ?? 10;
            const sideAlpha = opts.sideAlpha ?? .25;
            const meta = chart.getDatasetMeta(0);
            if (!meta || !meta.data?.length) return;
            if (!chart.options.cutout) chart.options.cutout = '55%';
            ctx.save();
            ctx.translate(0, depth);
            meta.data.forEach((arc, i) => {
                const vm = arc.getProps(['startAngle', 'endAngle', 'innerRadius', 'outerRadius', 'x', 'y'], true);
                const base = chart.data.datasets[0].backgroundColor;
                const color = Array.isArray(base) ? base[i % base.length] : base;
                const sideColor = withAlpha(color, sideAlpha);
                ctx.fillStyle = sideColor;
                ctx.beginPath();
                ctx.arc(vm.x, vm.y, vm.outerRadius, vm.startAngle, vm.endAngle);
                ctx.arc(vm.x, vm.y, vm.innerRadius, vm.endAngle, vm.startAngle, true);
                ctx.closePath();
                ctx.fill();
            });
            ctx.restore();
        }

        function applyBarGradient(chart) {
            const area = chart.chartArea;
            if (!area) return;
            const g = chart.ctx.createLinearGradient(0, area.top, 0, area.bottom);
            g.addColorStop(0, 'rgba(16,185,129,0.95)');
            g.addColorStop(1, 'rgba(5,150,105,0.85)');
            chart.data.datasets[0].backgroundColor = g;
            chart.data.datasets[0].borderRadius = 12;
            chart.update('none');
        }

        function applyPieColors(chart) {
            const n = chart.data.labels?.length || 0;
            chart.data.datasets[0].backgroundColor = genPalette(n);
            chart.options.cutout = chart.options.cutout || '55%';
            chart.update('none');
        }

        /* ===== Charts init ===== */
        const barCtx = document.getElementById('chartBar');
        const pieCtx = document.getElementById('chartPie');

        // ==== PALETA AUTOMÁTICA ====
        function genPalette(n) {
            const out = [];
            for (let i = 0; i < n; i++) {
                const hue = Math.round((360 * i) / Math.max(1, n));
                out.push(`hsl(${hue}deg 70% 55% / 0.9)`);
            }
            return out;
        }

        const palette = genPalette(labelsPHP.length);

        // ==== BARRAS (votos absolutos) ====
        const barChart = new Chart(barCtx, {
            type: 'bar',
            data: {
                labels: labelsPHP,
                datasets: [{
                    label: 'Votos',
                    data: dataPHP,
                    backgroundColor: palette,
                    borderWidth: 0,
                    borderRadius: 8
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: (ctx) => `${ctx.label}: ${ctx.formattedValue} votos`
                        }
                    }
                },
                scales: {
                    x: {
                        ticks: {
                            color: '#9ca3af',
                            autoSkip: false,
                            maxRotation: 60,
                            minRotation: 60
                        },
                        grid: {
                            display: false
                        }
                    },
                    y: {
                        ticks: {
                            color: '#9ca3af'
                        },
                        grid: {
                            color: 'rgba(255,255,255,0.05)'
                        },
                        beginAtZero: true
                    }
                }
            }
        });

        // ==== TORTA (composición %) ====
        const pieChart = new Chart(pieCtx, {
            type: 'doughnut',
            data: {
                labels: labelsPHP,
                datasets: [{
                    data: (() => {
                        const total = dataPHP.reduce((a, b) => a + Number(b || 0), 0);
                        return dataPHP.map(v => total ? (v * 100 / total) : 0);
                    })(),
                    backgroundColor: palette
                }]
            },
            options: {
                responsive: true,
                cutout: '55%',
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            color: '#e2e8f0',
                            boxWidth: 16
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: (ctx) => `${ctx.label}: ${ctx.parsed.toFixed(1)}%`
                        }
                    }
                }
            }
        });


        /* ===== Look ===== */
        (function makeButtons3D() {
            const selectors = ['button.bg-emerald-600', 'button.bg-slate-800', '#toggleTheme', '#toggleThemeTop', '#toggleVoz'];
            document.querySelectorAll(selectors.join(',')).forEach(b => b.classList.add('btn-3d'));
        })();

        /* ===== Utils ===== */
        function p2color(p) {
            const t = Math.max(0, Math.min(100, p || 0)) / 100;
            const hue = 40 - 40 * t;
            return `hsl(${hue}deg 60% 65% / 0.85)`;
        }

        function setLoading(b) {
            document.body.classList.toggle('loading', !!b);
            document.getElementById('barSkeleton').classList.toggle('hidden', !b);
            document.getElementById('pieSkeleton').classList.toggle('hidden', !b);
        }

        /* ===== Voz ===== */
        if (typeof window.vozActiva === 'undefined') window.vozActiva = true;
        document.getElementById('toggleVoz')?.addEventListener('click', () => {
            window.vozActiva = !window.vozActiva;
            const s = document.getElementById('toggleVozState');
            if (s) s.textContent = window.vozActiva ? 'ON' : 'OFF';
            if (!window.vozActiva && 'speechSynthesis' in window) {
                try {
                    window.speechSynthesis.cancel();
                } catch (e) {}
            }
        });



        // ==== NUEVO: estado de partido seleccionado (para título) ====
        let PARTIDO_ACTUAL = null;

        /* ===== Heatmap por Sección (mejorado) ===== */
        async function loadHeatmap() {
            const qs = new URLSearchParams(window.location.search);
            try {
                const res = await fetch('api_heatmap.php?' + qs.toString(), {
                    cache: 'no-store'
                });
                const json = await res.json();
                const wrap = document.getElementById('heatmapSecciones');
                wrap.innerHTML = '';

                const partidoId = Number(qs.get('partido_id') || 0);
                const tituloEl = document.getElementById('heatmapTitle');
                if (tituloEl) {
                    if (partidoId > 0 && PARTIDO_ACTUAL) {
                        tituloEl.textContent = `% votos de ${PARTIDO_ACTUAL} por Departamento`;
                    } else {
                        tituloEl.textContent = `%  Partido Ganador por Departamento`;
                    }
                }

                (json || []).forEach(row => {
                    const div = document.createElement('div');
                    div.className = 'rounded-xl p-3 border border-slate-800 soft';
                    const pct = Number(row.pct || 0);
                    div.style.background = p2color(pct);
                    const etiqueta = (partidoId > 0) ? `${pct.toFixed(1)}% del partido` : `${pct.toFixed(1)}% del 1º partido`;
                    div.title = `${row.seccion}: ${row.partido_nombre||'—'} (${etiqueta})`;
                    div.style.cursor = 'pointer';
                    div.innerHTML = `
                        <div class="text-xs muted">${row.seccion}</div>
                        <div class="text-base font-bold">${row.partido_nombre||'—'}</div>
                        <div class="text-sm opacity-80">${etiqueta}</div>
                    `;
                    div.addEventListener('click', () => {
                        const url = new URL(window.location.href);
                        url.searchParams.set('seccion_id', row.seccion_id);
                        url.searchParams.set('municipio_id', 0);
                        window.location.href = url.toString();
                    });
                    wrap.appendChild(div);
                });
            } catch (e) {
                console.warn('Heatmap error', e);
            }
        }

        /* ===== Drill-down (mejorado: fija partido_id y recarga heatmap) ===== */
        async function openDrill(partidoId, partidoNombre) {
            PARTIDO_ACTUAL = partidoNombre;

            // Seteo partido_id en la URL sin recargar
            const url = new URL(window.location.href);
            url.searchParams.set('partido_id', partidoId);
            window.history.replaceState({}, '', url);

            // Redibujar heatmap en modo "partido seleccionado"
            loadHeatmap();

            // Traer datos del drill
            const qs = new URLSearchParams(window.location.search);
            qs.set('partido_id', partidoId);
            try {
                const res = await fetch('api_drilldown.php?' + qs.toString());
                const {
                    circuitos = [], mesas = []
                } = await res.json();
                const $body = document.getElementById('drillBody');
                $body.innerHTML = `
                    <div class="mb-3 flex items-center justify-between">
                        <div><strong>Partido:</strong> ${partidoNombre}</div>
                        <button class="text-xs bg-slate-800 px-2 py-1 rounded" onclick="clearPartidoFilter()">Limpiar filtro</button>
                    </div>
                    <h4 class="font-semibold mb-2">Por circuito</h4>
                    <div class="overflow-x-auto mb-4">
                        <table class="min-w-full">
                            <thead><tr><th class="text-left py-1">Circuito</th><th class="text-left py-1">Votos</th></tr></thead>
                            <tbody>
                                ${circuitos.map(r=>`<tr class="border-t border-slate-800"><td class="py-1 pr-4">${r.circuito}</td><td class="py-1">${r.votos}</td></tr>`).join('')}
                            </tbody>
                        </table>
                    </div>
                    <h4 class="font-semibold mb-2">Por mesa</h4>
                    <div class="overflow-x-auto">
                        <table class="min-w-full">
                            <thead><tr><th class="text-left py-1">Mesa</th><th class="text-left py-1">Establecimiento</th><th class="text-left py-1">Votos</th></tr></thead>
                            <tbody>
                                ${mesas.map(r=>`<tr class="border-t border-slate-800"><td class="py-1 pr-4">${r.mesa}</td><td class="py-1 pr-4">${r.establecimiento}</td><td class="py-1">${r.votos}</td></tr>`).join('')}
                            </tbody>
                        </table>
                    </div>
                `;
                document.getElementById('drillModal').classList.remove('hidden');
                document.getElementById('drillModal').classList.add('flex');
            } catch (e) {
                console.warn('Drilldown error', e);
            }
        }

        function closeDrill() {
            document.getElementById('drillModal').classList.add('hidden');
            document.getElementById('drillModal').classList.remove('flex');
        }

        function clearPartidoFilter() {
            PARTIDO_ACTUAL = null;
            const url = new URL(window.location.href);
            url.searchParams.delete('partido_id');
            window.history.replaceState({}, '', url);
            loadHeatmap();
            closeDrill();
        }

        /* ===== SSE (Live) ===== */
        let lastSeen = 0;
        (function initSSE() {
            try {
                const qs = new URLSearchParams(window.location.search);
                const es = new EventSource('sse.php?' + qs.toString());
                es.addEventListener('ping', async (ev) => {
                    const {
                        last_id
                    } = JSON.parse(ev.data || '{}');
                    if (typeof last_id === 'number' && last_id > lastSeen) {
                        lastSeen = last_id;
                        document.getElementById('lastUpdate').innerText = new Date().toLocaleTimeString('es-AR');
                        setLoading(true);
                        await Promise.all([loadHeatmap(), refreshPartidosYResumen()]);
                        setLoading(false);
                    }
                });
                es.addEventListener('heartbeat', () => {});
                es.onerror = () => {
                    console.warn('SSE desconectado');
                };
                window.addEventListener('beforeunload', () => es.close());
            } catch (e) {
                console.warn('SSE no soportado', e);
            }
        })();

        /* ===== Noticias en vivo ===== */
        let newsTimer = null;
        async function loadNews(q = null) {
            const userQ = q ?? (document.getElementById('newsQuery')?.value || 'elecciones catamarca');
            const url = `news_rss.php?q=${encodeURIComponent(userQ)}&limit=12`;
            try {
                const r = await fetch(url, {
                    cache: 'no-store'
                });
                if (!r.ok) throw new Error('HTTP ' + r.status);
                const json = await r.json();

                // Lista
                const wrap = document.getElementById('newsList');
                wrap.innerHTML = (json.items || []).map(it => {
                    const dt = it.ts ? new Date(it.ts * 1000) : null;
                    const hh = dt ? dt.toLocaleTimeString('es-AR', {
                        hour: '2-digit',
                        minute: '2-digit'
                    }) : '';
                    const src = it.source ? `<span class="text-xs opacity-70"> · ${it.source}</span>` : '';
                    return `
                        <a href="${it.link}" target="_blank" rel="noopener" class="block p-3 rounded-xl bg-slate-900/60 hover:bg-slate-800 soft">
                            <div class="text-xs opacity-70">${hh}${src}</div>
                            <div class="font-medium mt-1 leading-snug">${it.title}</div>
                        </a>`;
                }).join('') || '<div class="opacity-70">Sin noticias por ahora</div>';

                // Ticker
                const ticker = document.getElementById('newsTickerInner');
                ticker.innerHTML = (json.items || []).slice(0, 8).map(it => `<span class="mx-4">• ${it.title}</span>`).join('');
                startTicker();

                document.getElementById('newsUpdated').innerText = new Date().toLocaleTimeString('es-AR');
            } catch (e) {
                console.warn('News error', e);
                document.getElementById('newsList').innerHTML = '<div class="opacity-70">No se pudo cargar noticias</div>';
            }
        }

        function startTicker() {
            const inner = document.getElementById('newsTickerInner');
            if (!inner) return;
            const parent = document.getElementById('newsTicker');
            let x = parent.offsetWidth;
            cancelAnimationFrame(startTicker._rafId);
            const step = () => {
                x -= 1;
                if (x + inner.offsetWidth < 0) x = parent.offsetWidth;
                inner.style.transform = `translateX(${x}px)`;
                startTicker._rafId = requestAnimationFrame(step);
            };
            step();
        }
        document.getElementById('newsBtn')?.addEventListener('click', () => loadNews());
        document.getElementById('newsQuery')?.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') loadNews();
        });

        function scheduleNews() {
            if (newsTimer) clearInterval(newsTimer);
            newsTimer = setInterval(() => loadNews(), 120000);
        }

        /* ===== REFRESH (datos + KPIs + charts) ===== */
        /* ===== REFRESH (datos + KPIs + charts) ===== */
        async function refreshPartidosYResumen() {
            const qs = new URLSearchParams(window.location.search);
            try {
                const [pRaw, resumen] = await Promise.all([
                    fetch('api_partidos.php?' + qs.toString()).then(r => r.json()),
                    fetch('api_resumen.php?' + qs.toString()).then(r => r.json()),
                ]);

                const partidos = [...(pRaw || [])].sort((a, b) => Number(b.votos || 0) - Number(a.votos || 0));

                // Partidos reales
                const labels = partidos.map(p => p.nombre);
                const data = partidos.map(p => Number(p.votos || 0));

                // Virtuales
                const blancos = Number(resumen.blancos || 0);
                const nulosObs = Number(resumen.nulos || 0) + Number(resumen.recurridos || 0) + Number(resumen.impugnados || 0);

                // Totales DINÁMICOS
                const total_validos = data.reduce((a, b) => a + Number(b || 0), 0); // solo partidos reales
                const total_general = total_validos + blancos + nulosObs; // partidos + blancos + nulos/obs

                // Barras (partidos + virtuales)
                // Barras (partidos + virtuales)
                const labelsFull = [...labels, 'Blancos', 'Nulos/Obs'];
                const dataFull = [...data, blancos, nulosObs];

                barChart.data.labels = labelsFull;
                barChart.data.datasets[0].data = dataFull;
                // 👇 VOLVER A COLORES POR BARRA
                barChart.data.datasets[0].backgroundColor = genPalette(labelsFull.length);
                barChart.update();

                // Pie: composición % sobre TOTAL GENERAL
                {
                    const dataPct = labelsFull.map((_, i) => total_general ? (Number(dataFull[i] || 0) * 100 / total_general) : 0);
                    pieChart.data.labels = labelsFull;
                    pieChart.data.datasets[0].data = dataPct;
                    pieChart.data.datasets[0].backgroundColor = genPalette(labelsFull.length);
                    pieChart.update();
                }

                // KPIs (top)
                const setIf = (id, v) => {
                    const el = document.getElementById(id);
                    if (el) el.innerText = v;
                };
                setIf('kpi-total', total_general.toLocaleString('es-AR'));
                setIf('kpi-votaron', Number(resumen.votaron || 0).toLocaleString('es-AR'));
                setIf('kpi-votaron2', Number(resumen.votaron || 0).toLocaleString('es-AR'));

                // KPIs secundarios
                const mesas_total = Number(resumen.mesas_total || 0);
                const mesas_cargadas = Number(resumen.mesas_cargadas || 0);
                const padron_total = Number(resumen.padron_total || 0);
                const participacion = (padron_total > 0) ? (Number(resumen.votaron || 0) * 100 / padron_total) : null;

                const kpiMesas = document.getElementById('kpi-mesas');
                const kpiMesasRest = document.getElementById('kpi-mesas-restantes');
                const kpiPart = document.getElementById('kpi-participacion');

                if (kpiMesas) kpiMesas.innerText = mesas_total > 0 ? ((mesas_cargadas * 100 / mesas_total).toFixed(1) + '%') : '—';
                if (kpiMesasRest) kpiMesasRest.innerText = mesas_total > 0 ? `Faltan ${(mesas_total - mesas_cargadas)} mesas` : '';
                if (kpiPart) kpiPart.innerText = participacion != null ? participacion.toFixed(1) + '%' : '—';

                // Duplicados en análisis
                setIf('kpi-mesas-dup', mesas_total > 0 ? (mesas_cargadas * 100 / mesas_total).toFixed(1) + '%' : '—');
                setIf('kpi-mesas-restantes-dup', mesas_total > 0 ? `Faltan ${(mesas_total - mesas_cargadas)} mesas` : '');
                setIf('kpi-participacion-dup', participacion != null ? participacion.toFixed(1) + '%' : '—');

                // Diferencia 1° vs 2° (sobre válidos)
                if (partidos.length >= 2) {
                    const p1 = Number(partidos[0].votos || 0);
                    const p2 = Number(partidos[1].votos || 0);
                    const pct1 = total_general ? (p1 * 100 / total_general) : 0;
                    const pct2 = total_general ? (p2 * 100 / total_general) : 0;
                    setIf('kpi-diferencia-dup', (pct1 - pct2).toFixed(1) + ' pts');
                }

                // Reaplicar look
                // applyBarGradient(barChart);
                applyPieColors(pieChart);

            } catch (e) {
                console.warn('refreshPartidosYResumen error', e);
            }
        }


        /* ===== Buscador en tabla ===== */
        document.getElementById('searchPartido').addEventListener('input', e => {
            const q = e.target.value.toLowerCase();
            document.querySelectorAll('#tablaPartidos tr').forEach(tr => {
                tr.style.display = tr.textContent.toLowerCase().includes(q) ? '' : 'none';
            });
        });

        /* ===== Export CSV ===== */
        function exportCSV() {
            const qs = new URLSearchParams(window.location.search);
            window.location.href = 'export_excel.php?' + qs.toString();
        }

        /* ===== Footer sync last update ===== */
        (function patchLastUpdateToFooter() {
            const elFooter = document.getElementById('lastUpdateFooter');
            const elTop = document.getElementById('lastUpdate');
            if (!elFooter || !elTop) return;
            const sync = () => {
                elFooter.textContent = elTop.textContent || '—';
            };
            sync();
            const observer = new MutationObserver(sync);
            observer.observe(elTop, {
                childList: true,
                subtree: true
            });
        })();

        /* ===== INIT ===== */
        (async () => {
            document.getElementById('lastUpdate').innerText = new Date().toLocaleTimeString('es-AR');
            setLoading(true);
            await Promise.all([loadHeatmap(), refreshPartidosYResumen()]);
            setLoading(false);

            // Noticias en vivo
            loadNews();
            scheduleNews();

            // Look 3D
            document.getElementById('chartBarWrap')?.classList.add('panel-3d');
            document.getElementById('chartPieWrap')?.classList.add('panel-3d');
            document.querySelectorAll('select, input[type="text"], input[type="search"], input:not([type]), textarea')
                .forEach(el => el.classList.add('input-3d'));
        })();


        function toggleChart(which) {
            const barBox = document.getElementById('chartsBar');
            const pieBox = document.getElementById('chartsPie');
            const bBar = document.getElementById('btnChartBar');
            const bPie = document.getElementById('btnChartPie');

            const showBar = (which === 'bar');

            // Mostrar/ocultar
            barBox.classList.toggle('hidden', !showBar);
            pieBox.classList.toggle('hidden', showBar);

            // Estado visual botones
            bBar.setAttribute('aria-pressed', showBar ? 'true' : 'false');
            bPie.setAttribute('aria-pressed', showBar ? 'false' : 'true');
            bBar.classList.toggle('bg-emerald-600/20', showBar);
            bBar.classList.toggle('hover:bg-emerald-600/30', showBar);
            bPie.classList.toggle('bg-emerald-600/20', !showBar);
            bPie.classList.toggle('hover:bg-emerald-600/30', !showBar);

            // 🔧 Importante: forzar re-cálculo del tamaño al gráfico visible
            if (showBar && typeof barChart !== 'undefined' && barChart) {
                requestAnimationFrame(() => {
                    barChart.resize();
                    barChart.update();
                });
            } else if (!showBar && typeof pieChart !== 'undefined' && pieChart) {
                requestAnimationFrame(() => {
                    pieChart.resize();
                    pieChart.update();
                });
            }
        }


        document.querySelector('form').addEventListener('submit', (e) => {
            const cargoText = document.querySelector('#cargo option:checked')?.textContent?.toUpperCase() || '';
            const requiereMun = cargoText.includes('INTENDENTE') || cargoText.includes('CONCEJAL');
            const mun = document.getElementById('municipio').value;
            if (requiereMun && (mun === null || mun === '0')) {
                e.preventDefault();
                alert('Para este cargo tenés que elegir un Municipio.');
            }
        });


        // Texto visible del option seleccionado
        function getSelectedText(sel) {
            const opt = sel.options[sel.selectedIndex];
            return (opt?.textContent || '').trim();
        }


        async function reloadMunicipios() {
            const secSelect = document.getElementById('seccion');
            const cargoSelect = document.getElementById('cargo');
            const elecSelect = document.getElementById('eleccion');
            const munSelect = document.getElementById('municipio');
            const msgBox = document.getElementById('noMunicipiosMsg');

            const secId = Number(secSelect?.value || 0);
            const cargoId = Number(cargoSelect?.value || 0);
            const elecId = Number(elecSelect?.value || 0);

            // limpiar
            munSelect.innerHTML = '<option value="0">Todos</option>';
            msgBox.textContent = '';
            msgBox.classList.add('hidden');

            if (secId <= 0) return; // Toda la provincia: no filtramos municipios

            try {
                const url = `municipios.php?seccion_id=${secId}&cargo_id=${cargoId}&eleccion_id=${elecId}`;
                const res = await fetch(url, {
                    cache: 'no-store'
                });
                const items = await res.json();

                (items || []).forEach(m => {
                    const opt = document.createElement('option');
                    opt.value = m.id;
                    opt.textContent = m.nombre;
                    munSelect.appendChild(opt);
                });

                if (!items || items.length === 0) {
                    const secName = secSelect.options[secSelect.selectedIndex]?.textContent?.trim() || 'ese departamento';
                    const cargoTxt = cargoSelect.options[cargoSelect.selectedIndex]?.textContent?.toLowerCase() || 'ese cargo';
                    msgBox.textContent = `El departamento ${secName} no vota ${cargoTxt}.`;
                    msgBox.classList.remove('hidden');
                }
            } catch (e) {
                console.warn('municipios.php error', e);
            }
        }

        // Dispara la recarga cuando cambian los selects
        ['seccion', 'cargo', 'eleccion'].forEach(id => {
            document.getElementById(id)?.addEventListener('change', reloadMunicipios);
        });

        // Primera carga (por si ya vienen elegidos)
        document.addEventListener('DOMContentLoaded', () => {
            reloadMunicipios();
        });

        // Validación: si el cargo requiere municipio, obligar selección
        document.getElementById('filtrosForm')?.addEventListener('submit', (e) => {
            const cargoText = document.querySelector('#cargo option:checked')?.textContent?.toUpperCase() || '';
            const requiereMun = cargoText.includes('INTENDENTE') || cargoText.includes('CONCEJAL');
            const munVal = document.getElementById('municipio')?.value;
            if (requiereMun && (!munVal || munVal === '0')) {
                e.preventDefault();
                alert('Para este cargo tenés que elegir un Municipio.');
            }
        });
    </script>
</body>

</html>